﻿using System.Collections.Generic;

namespace Hims.Api.Controllers
{
    using System;
    using System.Linq;
    using System.Threading.Tasks;
    using Domain.Configurations;
    using Domain.Helpers;
    using Domain.Services;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Models.InternalMedicine;
    using Newtonsoft.Json;
    using Newtonsoft.Json.Serialization;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.Library;
    using Shared.Library.Enums;
    using Shared.UserModels;
    using Shared.UserModels.Filters;
    using Utilities;
    using AppointmentCheck = Shared.UserModels.AppointmentCheck;
    using Hims.Api.Helper;
    using Hims.Domain.Entities;
    using Hims.Api.Senders;

    /// <summary>
    /// The internal medicine controller.
    /// </summary>
    [Route("api/internal-medicine")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class InternalMedicineController : BaseController
    {
        /// <summary>
        /// the provider service
        /// </summary>
        private readonly IEncounterService encounterServices;

        /// <summary>
        /// The push notification helper.
        /// </summary>
        private readonly IPushNotificationHelper pushNotificationHelper;

        /// <summary>
        /// The account session services.
        /// </summary>
        private readonly IAccountSessionService accountSessionServices;

        /// <summary>
        /// the provider service
        /// </summary>
        private readonly IInternalMedicineService internalMedicineService;

        /// <summary>
        /// the aes helper
        /// </summary>
        private readonly IAESHelper aesHelper;

        /// <summary>
        /// The appointments services.
        /// </summary>
        private readonly IAppointmentService appointmentsServices;

        /// <summary>
        /// The account service.
        /// </summary>
        private readonly IAccountService accountService;

        /// <summary>
        /// The amazon s3 helper.
        /// </summary>
        private readonly IDocumentHelper documentHelper;

        /// <summary>
        /// The configuration.
        /// </summary>
        private readonly IAmazonS3Configuration configuration;

        private readonly IAppointmentCheckService checkService;

        /// <inheritdoc />
        public InternalMedicineController(
            IInternalMedicineService internalMedicineService,
            IAppointmentService appointmentsServices,
            IAESHelper aesHelper,
            IDocumentHelper documentHelper,
            IEncounterService encounterServices,
            IPushNotificationHelper pushNotificationHelper,
            IAccountSessionService accountSessionServices,
            IAmazonS3Configuration configuration,
            IAccountService accountService,
            IAppointmentCheckService checkService
            )
        {
            this.internalMedicineService = internalMedicineService;
            this.appointmentsServices = appointmentsServices;
            this.aesHelper = aesHelper;
            this.documentHelper = documentHelper;
            this.configuration = configuration;
            this.accountService = accountService;
            this.encounterServices = encounterServices;
            this.pushNotificationHelper = pushNotificationHelper;
            this.accountSessionServices = accountSessionServices;
            this.checkService = checkService;
        }

        /// <summary>
        /// The find internal medicine.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("find")]
        [ProducesResponseType(typeof(EncounterResource), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FindAsync([FromBody] EncounterFilterModel model)
        {
            model = (EncounterFilterModel)EmptyFilter.Handler(model);
            var appointmentId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedAppointmentId));
            var internalMedicine = await this.internalMedicineService.FindAsync(appointmentId, model.IsAdmission);
            var providerId = !string.IsNullOrEmpty(model.EncryptedProviderId)
                ? Convert.ToInt32(this.aesHelper.Decode(model.EncryptedProviderId))
                : 0;
            var patientId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedPatientId));
            var appointmentList = new List<AppointmentModel>();
            var oldAppointment = new PreviousAppointmentModel();

            if (!model.IsAdmission)
            {
                var appointments = await this.appointmentsServices.FindByPatientAsync(patientId, providerId);
                appointmentList = appointments.ToList();

                foreach (var item in appointmentList)
                {
                    item.EncryptedAppointmentId = this.aesHelper.Encode(item.AppointmentId.ToString());
                    item.EncryptedPatientId = model.EncryptedPatientId;
                    item.AppointmentTimeString = Convert.ToDateTime(DateTime.Now.ToString("yyyy-MM-dd"))
                        .Add(item.AppointmentTime).ToString("hh:mm tt");
                }


                if (internalMedicine.EncounterId == null)
                {
                    oldAppointment = await this.appointmentsServices.FindPreviousAppointmentAsync(appointmentId);
                    if (oldAppointment != null)
                    {
                        oldAppointment.EncryptedAppointmentId =
                            this.aesHelper.Encode(oldAppointment.AppointmentId.ToString());
                    }
                }
            }

            return this.Success(new { Dashboard = internalMedicine, Appointments = appointmentList, OldAppointment = oldAppointment });
        }

        /// <summary>
        /// The find internal medicine.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("find-internal-medicine")]
        [ProducesResponseType(typeof(EncounterResource), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FindInternalMedicineAsync([FromBody] EncounterFilterModel model)
        {
            model = (EncounterFilterModel)EmptyFilter.Handler(model);
            var appointmentId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedAppointmentId));
            var internalMedicine = await this.internalMedicineService.FindAsync(appointmentId, model.IsAdmission);
            return this.Success(internalMedicine);
        }

        /// <summary>
        /// To find internal medicine dashboard
        /// </summary>
        /// <param name="model" >
        /// The encounter filter model.
        /// </param>
        /// <returns>
        /// The encounter model.
        /// </returns>
        [HttpPost]
        [Route("find-dashboard")]
        [ProducesResponseType(typeof(EncounterResource), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FindDashboardAsync([FromBody] EncounterFilterModel model)
        {
            model = (EncounterFilterModel)EmptyFilter.Handler(model);
            var appointmentId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedAppointmentId));
            var response = await this.internalMedicineService.FindDashboardAsync(appointmentId, model.Type, model.IsAdmission);
            if (response != null)
            {
                response.EncryptedAppointmentId = this.aesHelper.Encode(appointmentId.ToString());
            }
            return this.Success(response);
        }

        /// <summary>
        /// The modify async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("modify")]//// when its get called.
        [ProducesResponseType(typeof(int), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ModifyAsync([FromBody] EncounterModifyModel model)
        {
            model = (EncounterModifyModel)EmptyFilter.Handler(model);
            var encounterId = model.EncounterId == 0
                ? await this.internalMedicineService.AddAltAsync(model)
                : await this.internalMedicineService.UpdateAltAsync(model);
            switch (encounterId.Response)
            {
                case 0:
                    return this.ServerError();
                default:
                    var basicDetails = await this.encounterServices.GetBasicAppointmentDetails(model.AppointmentId, model.IsAdmission);
                    await this.appointmentsServices.UpdateEncounterTypeAsync(model.AppointmentId,
                        (int)EncounterTypes.InternalMedicine, model.IsAdmission);
                    //await this.appointmentsServices.AddProviderDrug(model.DrugId, model.DrugName, model.MedicineSearchType, model.ProviderId, model.ModifiedBy);

                    if (encounterId.Response > 0)
                    {

                        if (model.Type == InternalMedicine.Vitals && model.AddCheckPoint)
                        {
                            await this.checkService.InsertAsync(new AppointmentCheck.InsertModel
                            {
                                AppointmentId = model.AppointmentId,
                                Action = model.EncounterId == 0 ? "patient_vitals_added" : "patient_vitals_updated",
                                CreatedBy = model.ModifiedBy,
                                UserId = model.UserId
                            });
                        }

                        if (encounterId.Status == 1 && model.EncounterId == 0)
                        {
                            await NotificationHelper.Notification(
                                basicDetails.PatientId,
                                Roles.Patient,
                                model.EncounterId == 0
                                    ? NotificationIntimate.FullTranscriptionAdded
                                    : NotificationIntimate.FullTranscriptionUpdated,
                                this.aesHelper.Encrypt(model.AppointmentId.ToString()),
                                this.accountSessionServices,
                                this.pushNotificationHelper);
                        }

                        if (encounterId.Status == 2)
                        {
                            await NotificationHelper.Notification(
                                basicDetails.PatientId,
                                Roles.Patient,
                                model.EncounterId == 0
                                    ? NotificationIntimate.PrescriptionAdded
                                    : NotificationIntimate.PrescriptionUpdated,
                                this.aesHelper.Encrypt(model.AppointmentId.ToString()),
                                this.accountSessionServices,
                                this.pushNotificationHelper);
                        }
                    }
                    return this.Success(encounterId.Response);
            }
        }

        /// <summary>
        /// The find prescription async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [AllowAnonymous]
        [HttpPost]
        [Route("find-prescription")]
        [ProducesResponseType(typeof(InternalMedicinePrescriptionModel), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FindPrescriptionAsync([FromBody] EncounterFilterModel model)
        {
            model = (EncounterFilterModel)EmptyFilter.Handler(model);
            var appointmentId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedAppointmentId));
            var internalMedicinePrescription = await this.internalMedicineService.FindPrescriptionAsync(appointmentId);
            if (internalMedicinePrescription == null || internalMedicinePrescription.EncounterId == 0)
            {
                return this.BadRequest("Sorry! We don't have a encounter in the system.");
            }

            internalMedicinePrescription.AppointmentTimeString = Convert.ToDateTime(DateTime.Now.ToString("yyyy-MM-dd"))
                .Add(internalMedicinePrescription.AppointmentTime).ToString("hh:mm tt");
            return this.Success(internalMedicinePrescription);
        }

        /// <summary>
        /// The find full transcript async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [AllowAnonymous]
        [HttpPost]
        [Route("find-full-transcript")]
        [ProducesResponseType(typeof(InternalMedicineFullTranscriptModel), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FindFullTranscriptAsync([FromBody] EncounterFilterModel model)
        {
            model = (EncounterFilterModel)EmptyFilter.Handler(model);
            var appointmentId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedAppointmentId));
            var internalMedicineFullTranscript = await this.internalMedicineService.FindFullTranscriptAsync(appointmentId);
            if (internalMedicineFullTranscript == null || internalMedicineFullTranscript.EncounterId == 0)
            {
                return this.BadRequest("Sorry! We don't have a encounter in the system.");
            }

            var additionalData = await this.encounterServices.GetAdditionalProviderDataAsync(appointmentId);

            if (additionalData != null)
            {
                internalMedicineFullTranscript.Educations = additionalData.Educations;
                internalMedicineFullTranscript.ProviderName = additionalData.ProviderName;
                internalMedicineFullTranscript.ProviderNo = additionalData.ProviderNo;
                internalMedicineFullTranscript.SpecializationName = additionalData.SpecializationName;
                internalMedicineFullTranscript.PracticeName = additionalData.PracticeName;
                internalMedicineFullTranscript.PracticeLocation = additionalData.PracticeLocation;
                internalMedicineFullTranscript.Signature = additionalData.Signature;
                internalMedicineFullTranscript.ClinicPicture = additionalData.ClinicPicture;
            }

            internalMedicineFullTranscript.AppointmentTimeString = Convert
                .ToDateTime(DateTime.Now.ToString("yyyy-MM-dd")).Add(internalMedicineFullTranscript.AppointmentTime)
                .ToString("hh:mm tt");
            return this.Success(internalMedicineFullTranscript);
        }

        /// <summary>
        /// The save encounter images.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("save-skin")]
        public async Task<ActionResult> SaveEncounterImages([FromBody] EncounterImages model)
        {
            model = (EncounterImages)EmptyFilter.Handler(model);
            var appointmentId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedAppointmentId));
            var getPatientAccount = await this.internalMedicineService.GetAccountBasedOnAppointmentId(appointmentId, model.IsAdmission);
            var guid = await this.accountService.FindGuidAsync(getPatientAccount.PatientId, Roles.Patient);

            foreach (var image in model.Images)
            {
                if (image.Base64String.Contains(this.configuration.BucketURL) || image.Base64String.Contains("assets"))
                {
                    continue;
                }

                string appointment;

                try
                {
                    appointment = getPatientAccount.AppointmentDate.Add(getPatientAccount.AppointmentTime)
                        .ToString("yyyy-MM-dd_hh:mm_tt");
                }
                catch
                {
                    appointment = getPatientAccount.AppointmentNo;
                }

                var folderName = $@"internalMedicine/{appointment}/PhysicalExam/{model.Type}";
                var url = await this.documentHelper.UploadEncounterImagesAsync(image.Base64String, guid, folderName,
                    image.Type);
                image.Base64String = url;
            }

            var finalModel = new EncounterModifyModel
            {
                AppointmentId = appointmentId,
                EncounterId = model.EncounterId ?? 0,
                JsonString = JsonConvert.SerializeObject(model,
                    new JsonSerializerSettings { ContractResolver = new CamelCasePropertyNamesContractResolver() }),
                Type = model.Type,
                ModifiedBy = model.ModifiedBy
            };
            return await this.ModifyAsync(finalModel);
        }
    }
}
